<?php

namespace ACFCustomDatabaseTables\Controller;

use ACFCustomDatabaseTables\Contract\ControllerInterface;
use ACFCustomDatabaseTables\Service\License;
use ACFCustomDatabaseTables\Utils\View;
use ACFCustomDatabaseTables\Utils\Request;

/**
 * Class LicenseFormController
 * @package ACFCustomDatabaseTables\Controller
 */
class LicenseFormController implements ControllerInterface {

	const DEACTIVATE_KEY = 'acfcdt_license_deactivate';
	const ACTIVATE_KEY = 'acfcdt_license_activate';

	/** @var License */
	private $license;

	/**
	 * LicenseFormController constructor.
	 *
	 * @param License $license
	 */
	public function __construct( License $license ) {
		$this->license = $license;
	}

	public function init() {
		$this->license->init();
		add_action( 'admin_menu', [ $this, 'watch' ] );
		add_action( 'admin_notices', [ $this, 'temp_admin_messages' ] );
	}

	/**
	 * Keeps an eye out for triggered de/activation processes, checks were allowed to proceed, then actions requests.
	 */
	public function watch() {
		if ( $this->activation_triggered() and $this->license->verify_nonce() ) {
			$this->license->activate();
		}

		if ( $this->deactivation_triggered() and $this->license->verify_nonce() ) {
			$this->license->deactivate();
		}
	}

	/**
	 * @return bool
	 */
	private function activation_triggered() {
		return (bool) Request::get( self::ACTIVATE_KEY, null, 'post' );
	}

	/**
	 * @return bool
	 */
	private function deactivation_triggered() {
		return (bool) Request::get( self::DEACTIVATE_KEY, null, 'post' );
	}

	public function render() {
		return View::prepare( 'license-form', [
			'license' => $this->license->get(),
			'license_input_name' => $this->license->license_key(),
			'option_group' => $this->license->group(),
			'license_is_valid' => $this->license->is_valid(),
			'deactivate_input_name' => self::DEACTIVATE_KEY,
			'activate_input_name' => self::ACTIVATE_KEY,
			'nonce' => $this->license->get_nonce(),
		] );
	}

	/**
	 * Temporary admin notices handling until we properly set up our admin notifier object as a dep
	 */
	public function temp_admin_messages() {
		$activation = Request::get( 'acfcdt_activation', null, 'get' );

		if ( $activation === 'false' && $message = urldecode( Request::get( 'message', null, 'get' ) ) ) {
			?>
			<div class="error">
				<p><?php echo $message; ?></p>
			</div>
			<?php
		}
	}

}