<?php

class RenderForm_Gutenberg_Ajax {
	const NONCE = 'renderform_gutenberg_nonce';

	public static function register_actions() {
		add_action( 'wp_ajax_renderform_get_template_data', [ __CLASS__, 'get_template_data' ] );
		add_action( 'wp_ajax_renderform_render_image', [ __CLASS__, 'render_image' ] );
		add_action( 'wp_ajax_renderform_set_featured_image', [ __CLASS__, 'set_featured_image' ] );
	}

	private static function is_input_valid() {
		return isset( $_POST['_wpnonce'] ) && wp_verify_nonce( $_POST['_wpnonce'], self::NONCE ) &&
		       ( current_user_can( 'edit_posts' ) || current_user_can( 'edit_pages' ) ) && ! empty( $_POST['template_id'] );
	}

	/**
	 * @throws RenderForm_API_Exception
	 */
	public static function get_template_data() {
		if ( self::is_input_valid() ) {
			$template_id = sanitize_text_field( $_POST['template_id'] );
			$api         = new RenderForm_API();
			$response    = $api->get_template_data( $template_id );
			wp_send_json_success( $response );
		}
		wp_send_json_error( 'Bad request!' );
	}

	public static function render_image() {
		if ( self::is_input_valid() ) {
			$change_dictionary = [];
			if ( ! empty( $_POST['changes'] ) && is_array( $_POST['changes'] ) ) {
				foreach ( $_POST['changes'] as $change ) {
					if ( ! empty( $change['key'] ) && isset( $change['value'] ) ) {
						$key   = sanitize_text_field( $change['key'] );
						$value = sanitize_text_field( $change['value'] );
						if ( ! empty( $key ) ) {
							$change_dictionary[ $change['key'] ] = $change['value'];
						}
					}
				}
			}
			$template_id = sanitize_text_field( $_POST['template_id'] );
			$api         = new RenderForm_API();
			try {
				wp_send_json_success( $api->get_rendered_image( $template_id, $change_dictionary ) );
			} catch ( RenderForm_API_Exception $e ) {
				wp_send_json_error( 'Bad response!' );
			}
		}
		wp_send_json_error( 'Bad request!' );
	}

	public static function set_featured_image() {
		if ( self::is_input_valid() ) {
			$post_id = absint( $_POST['post_id'] ?? 0 );
			if ( $post_id === 0 ) {
				wp_send_json_error( 'Post ID is required.' );
			}
			if ( empty( $_POST['image'] ) ) {
				wp_send_json_error( 'Image cannot be empty.' );
			}
			$image_url = $_POST['image'];
			if ( filter_var( $image_url, FILTER_VALIDATE_URL ) === false ) {
				wp_send_json_error( 'Image URL is not valid' );
			}
			if ( str_starts_with( $image_url, 'https://cdn.renderform.io/' ) ) {
				$image = media_sideload_image( $image_url, $post_id, '', 'id' );
				if ( $image ) {
					set_post_thumbnail( $post_id, $image );
					wp_send_json_success($image);
				} else {
					wp_send_json_error( $image);
				}
			}
		}
		wp_send_json_error( 'Bad request!' );
	}

}