<?php

/**
 * RenderForm_API class is used to communicate with RenderForm REST API
 */
final class RenderForm_API {
	private static $endpoints = [
		'template_data'  => 'https://api.renderform.io/api/v1/templates/:id',
		'templates_list' => 'https://api.renderform.io/api/v1/my-templates',
		'image'          => 'https://get.renderform.io/api/v1/render',
		'test_key'       => 'https://api.renderform.io/api/v1/my-templates'
	];

	private string $key;
	private array $templates;

	/**
	 * @throws RenderForm_API_Exception
	 */
	function __construct() {
		$this->key = RenderForm_Settings::get_apikey();
		if ( empty( $this->key ) ) {
			throw new RenderForm_API_Exception( 'RenderFrom API Key is not defined!' );
		}
	}

	/**
	 * Returns array of Http Request Headers
	 *
	 * @return array
	 */
	private function get_headers(): array {
		return [
			'Content-Type' => 'application/json',
			'X-API-KEY'    => sanitize_text_field( $this->key )
		];
	}

	private function get_endpoint( $key ): string {
		return RenderForm_API::$endpoints[ $key ] ?? false;
	}

	/**
	 * Makes remote POST/GET request to the API
	 *
	 * @param string $endpoint_key
	 * @param string $method
	 * @param array $query_parameters
	 * @param array $body
	 *
	 * @return array
	 * @throws RenderForm_API_Exception
	 */
	private function request( string $endpoint_key, string $method = 'GET', array $query_parameters = array(), array $body = array() ): array {
		$endpoint_url = $this->get_endpoint( $endpoint_key );
		if ( ! $endpoint_url ) {
			throw new RenderForm_API_Exception( 'Undefined endpoint URL' );
		}

		if ( $query_parameters ) {
			$endpoint_url = str_replace( array_keys( $query_parameters ), $query_parameters, $endpoint_url );
		}

		$res = wp_remote_request( $endpoint_url, [
			'method'  => $method,
			'headers' => $this->get_headers(),
			'body'    => ! empty( $body ) ? json_encode( $body, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_SLASHES ) : null,
			'timeout' => 45
		] );

		if ( ! is_wp_error( $res ) && $res['response']['code'] === 200 ) {
			return json_decode( $res['body'], true ) ?? array();
		}
		throw new RenderForm_API_Exception( 'Bad response!' );
	}

	/**
	 * Get template identifiers
	 *
	 * @return array
	 * @throws RenderForm_API_Exception
	 */
	public function get_templates(): array {
		$this->templates = array();
		foreach ( $this->request( 'templates_list' ) as $template ) {
			$this->templates[] = [
				'id'   => $template['identifier'],
				'name' => $template['name']
			];
		}

		return $this->templates;
	}

	/**
	 * Check if template id is valid
	 *
	 * @param $template_id
	 *
	 * @return bool
	 * @throws RenderForm_API_Exception
	 */
	public function is_template_id_valid( $template_id ) {
		if ( empty( $this->templates ) ) {
			$this->get_templates();
		}
		foreach ( $this->templates as $template ) {
			if ( $template['id'] === $template_id ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Gets template's detailed data from API
	 *
	 * @param $id
	 *
	 * @return array
	 * @throws RenderForm_API_Exception
	 */
	public function get_template_data( $id ): array {
		$template_data = $this->request( 'template_data', 'GET', [ ':id' => $id ] );

		$placeholders = $template_data['placeholders'];
		$result       = array();

		if ( $placeholders && is_array( $placeholders ) ) {
			foreach ( $placeholders as $item ) {
				if ( $this->is_property_required( $item ) ) {
					$result[] = [
						'key'          => $item['key'],
						'type'         => strtolower( $item['componentType'] ),
						'defaultValue' => $item['defaultValue']
					];
				}
			}
		}

		return $result;
	}

	/**
	 * Check if placeholder is required by the Plugin
	 *
	 * @param $item
	 *
	 * @return bool
	 */
	private function is_property_required( $item ): bool {
		return in_array(
			$item['property'],
			[ 'text', 'src' ]
		);
	}

	/**
	 * Checks if API KEY is valid by making GET request
	 *
	 * @param $key
	 *
	 * @return bool
	 */
	public static function is_api_key_valid( $key ): bool {
		$res = wp_remote_get( RenderForm_API::$endpoints['test_key'], [
			'headers' => [
				'Content-Type' => 'application/json',
				'X-API-KEY'    => sanitize_text_field( $key )
			]
		] );

		if ( ! is_wp_error( $res ) ) {
			return (int) $res['response']['code'] === 200;
		}

		return false;
	}

	/**
	 * Get rendered image url from API
	 *
	 * @param $template_id
	 * @param $changes_dictionary
	 *
	 * @return string
	 * @throws RenderForm_API_Exception
	 */
	public function get_rendered_image( $template_id, $changes_dictionary ): string {
		$request_body['template'] = $template_id;
		if ( ! empty( $changes_dictionary ) && is_array( $changes_dictionary ) ) {
			$request_body['changesDictionary'] = $changes_dictionary;
		}
		$response = $this->request( 'image', 'POST', [], $request_body );
		if ( ! empty( $response['href'] ) ) {
			return $response['href'];
		}
		throw new RenderForm_API_Exception( 'Bad response!' );
	}
}