<?php

/**
 * RenderForm_Settings Class adds plugin's admin page.
 */
class RenderForm_Settings {
	const SLUG_PAGE = 'renderform_settings';
	const SLUG_OPTIONS = 'renderform_options';
	const PREFIX = 'renderform_';
	const PASSWORD_PLACEHOLDER = '########################';

	/**
	 * @var array settings schema
	 */
	private static array $fields_schema;

	/**
	 * Register hooks
	 *
	 * @return void
	 */
	public static function init(): void {
		add_action( 'admin_init', [ __CLASS__, 'register_settings' ] );
		add_action( 'admin_menu', [ __CLASS__, 'add_settings_menu_page' ] );
	}

	/**
	 * @return void
	 */
	private static function define_fields_schema() {
		self::$fields_schema = [
			'apikey' => [
				'label'         => __( 'API Key', 'renderform' ),
				'description'   => '',
				'type'          => 'password',
				'default_value' => ''
			]
		];
	}

	/**
	 * Validates and sanitizes user input
	 *
	 * @param array $input
	 *
	 * @return array
	 */
	public static function validate( $input ): array {
		if ( $input['apikey'] === self::PASSWORD_PLACEHOLDER ) {
			$input['apikey'] = self::get_apikey();
		} else {
			$input['apikey'] = sanitize_text_field( $input['apikey'] );
		}

		return $input;
	}

	/**
	 * Registers the Wordpress Settings API, adds sections and settings fields
	 *
	 * @return void
	 */
	public static function register_settings() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		self::define_fields_schema();
		register_setting( self::SLUG_PAGE, self::SLUG_OPTIONS, [ __CLASS__, 'validate' ] );

		$section = 'renderform_section_general';
		add_settings_section(
			$section,
			'',
			null,
			self::SLUG_PAGE
		);

		foreach ( self::$fields_schema as $name => $field ) {
			add_settings_field(
				self::PREFIX . $name,
				$field['label'],
				[ __CLASS__, 'render_html_field' ],
				self::SLUG_PAGE,
				$section,
				[
					'name'          => $name,
					'type'          => $field['type'],
					'default_value' => $field['default_value'],
					'description'   => $field['description']
				]
			);
		}
	}

	/**
	 * Adds new settings page to the WP Admin menu
	 *
	 * @return void
	 */
	public static function add_settings_menu_page() {
		add_submenu_page(
			'options-general.php',
			'RenderForm',
			'RenderForm',
			'manage_options',
			self::SLUG_PAGE,
			[ __CLASS__, 'render_html_page_content' ]
		);
	}

	/**
	 * Prints HTML code of the settings page
	 *
	 * @return void
	 */
	public static function render_html_page_content() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		if ( isset( $_GET['settings-updated'] ) ) {
			$is_api_key_valid = RenderForm_API::is_api_key_valid( RenderForm_Settings::get_apikey() );
			if ( $is_api_key_valid ) {
				add_settings_error( self::PREFIX . 'messages', self::PREFIX . 'message', __( 'The given API Key is valid!', 'renderform' ), 'success' );
			} else {
				add_settings_error( self::PREFIX . 'messages', self::PREFIX . 'message', __( 'The given API Key is not valid!', 'renderform' ) );
			}
			settings_errors( self::PREFIX . 'messages' );
		}
		?>
        <div class="wrap">
            <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
            <form action="options.php" method="post">
				<?php
				settings_fields( self::SLUG_PAGE );
				do_settings_sections( self::SLUG_PAGE );
				submit_button( __( 'Save settings', 'renderform' ) );
				?>
            </form>
        </div>
		<?php
	}

	/**
	 * Prints HTML code of the single field
	 *
	 * @param $args
	 *
	 * @return void
	 */
	public static function render_html_field( $args ) {
		$field_name = $args['name'];
		$field_type = $args['type'];
		$value      = get_option( self::SLUG_OPTIONS );

		$value = $value[ $field_name ] ?? $args['default_value'];
		$name  = self::SLUG_OPTIONS . '[' . $field_name . ']';

		if ( $field_type === 'password' ) {
			echo '<input type="' . esc_attr( $field_type ) . '" class="regular-text" name="' . esc_attr( $name ) . '" value="' . ( ! empty( $value ) ? self::PASSWORD_PLACEHOLDER : '' ) . '" />';
		}
	}

	/**
	 * Returns value of the setting field
	 *
	 * @param $field_key
	 *
	 * @return mixed
	 */
	public static function get_value( $field_key ) {
		$options = get_option( self::SLUG_OPTIONS );
		if ( isset( $options[ $field_key ] ) ) {
			return $options[ $field_key ];
		}

		return self::$fields_schema[ $field_key ]['default_value'];
	}

	public static function get_apikey(): string {
		return self::get_value( 'apikey' );
	}

    public static function get_admin_url(): string {
        return admin_url('options-general.php?page=' . self::SLUG_PAGE);
    }
}