<?php
/**
 * Utility class for WooCommerce-related operations.
 *
 * This class serves as a container for utility methods and functionality
 * specific to interacting with WooCommerce.
 *
 * @since ??
 *
 * @package Divi
 */

namespace ET\Builder\Packages\WooCommerce;

use ET\Builder\Framework\Utility\ArrayUtility;
use ET\Builder\Framework\Utility\Conditions;
use ET\Builder\FrontEnd\Assets\DynamicAssets;
use ET\Builder\FrontEnd\Assets\DynamicAssetsUtils;
use ET_Theme_Builder_Layout;
use ET_Theme_Builder_Woocommerce_Product_Variable_Placeholder;
use stdClass;
use WP_Query;

if ( ! defined( 'ABSPATH' ) ) {
	die( 'Direct access forbidden.' );
}


/**
 * Utility class for various WooCommerce-related operations and helper functions.
 */
class WooCommerceUtils {
	/**
	 * An array of allowed WooCommerce functions.
	 *
	 * @since ??
	 *
	 * @var string[]
	 */
	private static $_allowed_functions = [
		'the_title',
		'woocommerce_breadcrumb',
		'woocommerce_template_single_price',
		'woocommerce_template_single_add_to_cart',
		'woocommerce_product_additional_information_tab',
		'woocommerce_template_single_meta',
		'woocommerce_template_single_rating',
		'woocommerce_show_product_images',
		'wc_get_stock_html',
		'wc_print_notices',
		'wc_print_notice',
		'woocommerce_output_related_products',
		'woocommerce_upsell_display',
		'woocommerce_checkout_login_form',
		'wc_cart_empty_template',
		'woocommerce_output_all_notices',
	];

	/**
	 * Check if current global $post uses builder / layout block, not `product` CPT, and contains
	 * WooCommerce module inside it. This check is needed because WooCommerce by default only adds
	 * scripts and style to `product` CPT while WooCommerce Modules can be used at any CPT.
	 *
	 * Based on legacy `et_builder_wc_is_non_product_post_type` function.
	 *
	 * @since ??
	 *
	 * @return bool
	 */
	public static function is_non_product_post_type(): bool {
		static $is_non_product_post_type;

		// If the result is already cached, return it immediately.
		if ( null !== $is_non_product_post_type ) {
			return $is_non_product_post_type;
		}

		// Bail early for specific request types (e.g., AJAX requests, REST API requests, or VB top window requests).
		if ( Conditions::is_ajax_request() || Conditions::is_rest_api_request() || Conditions::is_vb_top_window() ) {
			$is_non_product_post_type = false;
			return $is_non_product_post_type;
		}

		global $post;

		// If the global $post is missing or is a WooCommerce 'product', immediately return false.
		if ( empty( $post ) || 'product' === $post->post_type ) {
			$is_non_product_post_type = false;
			return $is_non_product_post_type;
		}

		// Skip further checks if builder or layout block isn't used.
		$is_builder_used      = et_pb_is_pagebuilder_used( $post->ID );
		$is_layout_block_used = has_block( 'divi/layout', $post->post_content );

		if ( ! $is_builder_used && ! $is_layout_block_used ) {
			$is_non_product_post_type = false;
			return $is_non_product_post_type;
		}

		// Check if WooCommerce module is used in the post content.
		$has_wc_module = DynamicAssets::get_instance()->has_wc_shortcode();

		// Set the result based on the above checks.
		$is_non_product_post_type = ( $is_builder_used || $is_layout_block_used ) && $has_wc_module;

		return $is_non_product_post_type;
	}

	/**
	 * Returns TRUE if the Product attribute value is valid.
	 *
	 * Valid values are Product Ids, `current` and `latest`.
	 *
	 * @since ??
	 *
	 * @param string $maybe_product_id Product ID.
	 *
	 * @return bool
	 */
	public static function is_product_attr_valid( string $maybe_product_id ): bool {
		if ( empty( $maybe_product_id ) ) {
			return false;
		}

		if (
			absint( $maybe_product_id ) === 0
			&& ! in_array( $maybe_product_id, [ 'current', 'latest' ], true )
		) {
			return false;
		}

		return true;
	}

	/**
	 * Retrieves the default column settings for WooCommerce posts.
	 *
	 * This method applies the 'divi_woocommerce_get_default_columns' filter
	 * to determine and return the default column configuration.
	 *
	 * Based on the legacy `get_columns_posts_default` function.
	 *
	 * @since ??
	 *
	 * @return string The default column configuration for WooCommerce posts, as filtered by the applied hook.
	 */
	public static function get_default_columns_posts(): string {
		// Get the value for columns.
		$columns = self::get_default_columns_posts_value();

		/**
		 * Filters the default column configuration for WooCommerce posts.
		 *
		 * @since ??
		 *
		 * @param string $columns The default column configuration for WooCommerce posts.
		 */
		return apply_filters( 'divi_woocommerce_get_default_columns', $columns );
	}

	/**
	 * Retrieves the default number of columns for displaying posts.
	 *
	 * Determines the appropriate default column value based on the current page's
	 * layout and context. If the page has a sidebar, it returns a value indicative
	 * of a layout with a sidebar; otherwise, it defaults to standard values often
	 * influenced by WooCommerce settings.
	 *
	 * Based on the legacy `get_columns_posts_default_value` function.
	 *
	 * @since ??
	 *
	 * @return string The number of columns as a string. Returns '3' for layouts
	 *                with a sidebar or '4' as the default value.
	 */
	public static function get_default_columns_posts_value(): string {
		$post_id = et_core_page_resource_get_the_ID();

		// phpcs:ignore WordPress.Security.NonceVerification -- Nonce verification is not required.
		$post_id = (int) $post_id ? $post_id : ArrayUtility::get_value( $_POST, 'current_page.id' );

		$page_layout = get_post_meta( $post_id, '_et_pb_page_layout', true );

		if ( $page_layout && 'et_full_width_page' !== $page_layout && ! ET_Theme_Builder_Layout::is_theme_builder_layout() ) {
			return '3'; // Set to 3 if page has sidebar.
		}

		/*
		* Default number is based on the WooCommerce plugin default value.
		*
		* @see woocommerce_output_related_products()
		*/
		return '4';
	}

	/**
	 * Retrieves the default product configuration.
	 *
	 * Applies a filter to allow the retrieval or modification of the default product configuration.
	 * The returned value is expected to be an associative array containing details about the default product.
	 *
	 * Based on the legacy `get_product_default` function.
	 *
	 * @since ??
	 *
	 * @return string The default product configuration. The structure and content of the array
	 *               depend on the filter `divi_woocommerce_get_default_product`.
	 */
	public static function get_default_product(): string {
		// Get the value for the $default_product.
		$default_product = self::get_default_product_value();

		/**
		 * Filters the default product configuration.
		 *
		 * @since ??
		 *
		 * @param string $default_product The default product configuration.
		 */
		return apply_filters( 'divi_woocommerce_get_default_product', $default_product );
	}

	/**
	 * Retrieves the default product value identifier.
	 *
	 * Determines the default product value based on the current context, including post type
	 * or page resource. This method assesses whether the current post type is a "product"
	 * or a "theme builder layout" and returns an appropriate default value.
	 *
	 * Based on the legacy `get_product_default_value` function.
	 *
	 * @since ??
	 *
	 * @return string The default product value, either 'current' if the context relates to
	 *                a product or theme builder layout, or 'latest' as a fallback.
	 */
	public static function get_default_product_value(): string {
		$post_id = Conditions::is_rest_api_request()
			? get_the_ID()
			: et_core_page_resource_get_the_ID();

		// phpcs:ignore WordPress.Security.NonceVerification -- Nonce verification is not required.
		$post_id   = (int) $post_id ? $post_id : ArrayUtility::get_value( $_POST, 'current_page.id' );
		$post_type = get_post_type( $post_id );

		if ( 'product' === $post_type || et_theme_builder_is_layout_post_type( $post_type ) ) {
			return 'current';
		}

		return 'latest';
	}

	/**
	 * Retrieves the default WooCommerce tabs.
	 *
	 * This method returns the default WooCommerce tabs, allowing filters
	 * to modify the data before it is returned. It is primarily used
	 * to obtain the currently configured set of WooCommerce product tabs.
	 *
	 * Based on the legacy `get_woo_default_tabs` function.
	 *
	 * @since ??
	 *
	 * @return array The default WooCommerce tabs after applying filters.
	 */
	public static function get_default_product_tabs(): array {
		// Get the value for the $default_tabs.
		$default_tabs = self::get_default_product_tabs_options();

		/**
		 * Filters the default WooCommerce tabs.
		 *
		 * @since ??
		 *
		 * @param array $default_tabs The default WooCommerce tabs.
		 */
		return apply_filters( 'divi_woocommerce_get_default_product_tabs', $default_tabs );
	}

	/**
	 * Retrieves default WooCommerce product tabs options.
	 *
	 * Processes the current product data, applies necessary filters, and returns
	 * a list of available WooCommerce product tabs. Handles resetting global variables
	 * after usage to maintain consistent behavior.
	 *
	 * Based on the legacy `get_woo_default_tabs_options` function.
	 *
	 * @since ??
	 *
	 * @return array Array of default WooCommerce product tabs. Returns an empty array
	 *               if no valid tabs are found, or if the current product cannot be retrieved.
	 */
	public static function get_default_product_tabs_options(): array {
		// Bail if WooCommerce is not enabled.
		if ( ! function_exists( 'wc_get_product' ) ) {
			return [];
		}

		$maybe_product_id = self::get_default_product_value();
		$product_id       = self::get_product( $maybe_product_id );

		$current_product = wc_get_product( $product_id );
		if ( ! $current_product ) {
			return [];
		}

		global $product, $post;
		$original_product = $product;
		$original_post    = $post;
		$product          = $current_product;

		// phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited -- Intentionally done.
		$post = get_post( $product->get_id() );

		$tabs = apply_filters( 'woocommerce_product_tabs', [] );

		// Reset global $product.
		$product = $original_product;

		// phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited -- Intentionally done.
		$post = $original_post;

		if ( ! empty( $tabs ) ) {
			return array_keys( $tabs );
		}

		return [];
	}

	/**
	 * Retrieves the WooCommerce product tabs.
	 *
	 * This method fetches the available WooCommerce product tabs by invoking the
	 * 'woocommerce_product_tabs' filter. It ensures that appropriate product context
	 * is set globally in cases where it is not already defined. If no valid product
	 * context can be established, default product tab options are returned.
	 *
	 * Based on the legacy `et_fb_woocommerce_tabs` function.
	 *
	 * @since ??
	 *
	 * @return array An associative array of product tabs, where each key is the tab name,
	 *               and each value is an array containing 'value' (tab's name) and
	 *               'label' (tab's title).
	 */
	public static function get_product_tabs_options(): array {
		global $product, $post;

		$old_product = $product;
		$old_post    = $post;
		$is_product  = isset( $product ) && is_a( $product, 'WC_Product' );

		if ( ! $is_product && Conditions::is_woocommerce_enabled() ) {
			$product = self::get_product( 'latest' );

			if ( $product ) {
				// phpcs:ignore WordPress.WP.GlobalVariablesOverride -- Overriding global post is safe as the original $ post has been restored at the end.
				$post = get_post( $product->get_id() );
			} else {
				$product = $old_product;
				return self::set_default_product_tabs_options();
			}
		}

		// On non-product post-types, the filter will cause a fatal error unless we have a global $product set.
		$tabs    = apply_filters( 'woocommerce_product_tabs', [] );
		$options = array();

		foreach ( $tabs as $name => $tab ) {
			$options[ $name ] = array(
				'value' => $name,
				'label' => $tab['title'],
			);
		}

		// Reset global $product.
		$product = $old_product;

		// phpcs:ignore WordPress.WP.GlobalVariablesOverride -- Restoring original global $post data.
		$post = $old_post;

		return $options;
	}

	/**
	 * Retrieves the default page type configuration.
	 *
	 * Provides an array of default page type settings, which can be filtered
	 * by the 'divi_woocommerce_get_default_page_type' filter.
	 *
	 * Based on the legacy `get_page_type_default` function.
	 *
	 * @since ??
	 *
	 * @return string The default page type configuration.
	 */
	public static function get_default_page_type(): string {
		// Get the value for the $default_page_type.
		$default_page_type = self::get_default_page_type_value();

		/**
		 * Filters the default page type configuration.
		 *
		 * @since ??
		 *
		 * @param string $default_page_type The default page type configuration.
		 */
		return apply_filters( 'divi_woocommerce_get_default_page_type', $default_page_type );
	}

	/**
	 * Retrieves the default page type value based on the current page.
	 *
	 * Determines the page type by checking if the current page is a cart or checkout page.
	 * If neither condition is met, it defaults to the "product" page type.
	 *
	 * Based on the legacy `get_page_type_default_value` function.
	 *
	 * @since ??
	 *
	 * @return string The determined page type, which can be "cart", "checkout", or "product".
	 */
	public static function get_default_page_type_value(): string {
		$is_cart_page     = function_exists( 'is_cart' ) && is_cart();
		$is_checkout_page = function_exists( 'is_checkout' ) && is_checkout();

		if ( $is_cart_page ) {
			return 'cart';
		} elseif ( $is_checkout_page ) {
			return 'checkout';
		} else {
			return 'product';
		}
	}

	/**
	 * Retrieves the product ID based on the provided product attribute.
	 *
	 * Determines the correct product ID to return based on the given attribute,
	 * handling cases like "current", "latest", and numeric values. If the attribute
	 * is invalid, fallback mechanisms are employed to retrieve a relevant product ID.
	 *
	 * @since ??
	 *
	 * @param string $valid_product_attr The input attribute used to determine the product ID.
	 *                                   Acceptable values include "current", "latest",
	 *                                   or a numeric product ID.
	 *
	 * @return int The determined product ID. Returns 0 if no valid product ID can be resolved.
	 */
	public static function get_product_id_by_prop( string $valid_product_attr ): int {
		if ( ! self::is_product_attr_valid( $valid_product_attr ) ) {
			return 0;
		}

		if ( 'current' === $valid_product_attr ) {
			$current_post_id = DynamicAssetsUtils::get_current_post_id();

			if ( et_theme_builder_is_layout_post_type( get_post_type( $current_post_id ) ) ) {
				// We want to use the latest product when we are editing a TB layout.
				$valid_product_attr = 'latest';
			}
		}

		if (
			! in_array( $valid_product_attr, [ 'current', 'latest' ], true )
			&& false === get_post_status( $valid_product_attr )
		) {
			$valid_product_attr = 'latest';
		}

		if ( 'current' === $valid_product_attr ) {
			$product_id = DynamicAssetsUtils::get_current_post_id();
		} elseif ( 'latest' === $valid_product_attr ) {
			$args = [
				'limit'       => 1,
				'post_status' => [ 'publish', 'private' ],
				'perm'        => 'readable',
			];

			if ( ! function_exists( 'wc_get_products' ) ) {
				return 0;
			}

			$products = wc_get_products( $args );

			if ( isset( $product ) && is_a( $product, 'WC_Product' ) ) {
				$product_id = $products[0]->get_id();
			} else {
				return 0;
			}
		} elseif ( is_numeric( $valid_product_attr ) && 'product' !== get_post_type( $valid_product_attr ) ) {
			// There is a condition that $valid_product_attr value passed here is not the product ID.
			// For example when you set product breadcrumb as Blurb Title when building layout in TB.
			// So we get the most recent product ID in date descending order.
			$query = new \WC_Product_Query(
				[
					'limit'   => 1,
					'orderby' => 'date',
					'order'   => 'DESC',
					'return'  => 'ids',
					'status'  => [ 'publish' ],
				]
			);

			$products = $query->get_products();

			if ( $products && ! empty( $products[0] ) ) {
				$product_id = absint( $products[0] );
			} else {
				$product_id = absint( $valid_product_attr );
			}
		} else {
			$product_id = absint( $valid_product_attr );
		}

		return $product_id;
	}

	/**
	 * Retrieves the product object based on the provided product identifier.
	 *
	 * Resolves the WooCommerce product object corresponding to the given product identifier.
	 * Utilizes a helper method to determine the appropriate product ID and fetches the product
	 * if it exists. Returns false if no valid product can be retrieved.
	 *
	 * @since ??
	 *
	 * @param string $maybe_product_id The input value which may represent a product ID,
	 *                                 or another attribute to determine the product.
	 *
	 * @return \WC_Product|false The WooCommerce product object if successfully resolved,
	 *                           or false if no valid product can be found.
	 */
	public static function get_product( string $maybe_product_id ) {
		$product_id = self::get_product_id_by_prop( $maybe_product_id );

		if ( ! function_exists( 'wc_get_product' ) ) {
			return false;
		}

		$product = wc_get_product( $product_id );

		if ( empty( $product ) ) {
			return false;
		}

		return $product;
	}

	/**
	 * Retrieves the product ID from a given input.
	 *
	 * Resolves and returns the product ID associated with the input. If the input does not
	 * correspond to a valid product, the function will return 0.
	 *
	 * @since ??
	 *
	 * @param string $maybe_product_id A potential product identifier that will be validated
	 *                                 and used to retrieve the corresponding product ID.
	 *
	 * @return int The ID of the resolved product. Returns 0 if the input does not correspond
	 *             to a valid product.
	 */
	public static function get_product_id( string $maybe_product_id ): int {
		$product = self::get_product( $maybe_product_id );
		if ( ! $product ) {
			return 0;
		}

		return $product->get_id();
	}

	/**
	 * Retrieves the product ID based on the provided attributes.
	 *
	 * Determines the appropriate product ID by evaluating the provided arguments,
	 * handling cases such as "latest", "current", or a specific product ID.
	 * If the provided product ID is not valid or does not exist, a fallback mechanism
	 * is used to retrieve the latest product ID.
	 *
	 * @since ??
	 *
	 * @param array $args The input arguments containing product-related attributes.
	 *                    The "product" key may have a value of "latest", "current",
	 *                    or a specific product ID.
	 *
	 * @return int The resolved product ID. Returns 0 if no valid product ID can be determined.
	 */
	public static function get_product_id_from_attributes( array $args ): int {
		$maybe_product_id        = ArrayUtility::get_value( $args, 'product', 'latest' );
		$is_latest_product       = 'latest' === $maybe_product_id;
		$is_current_product_page = 'current' === $maybe_product_id;

		if ( $is_latest_product ) {
			// Dynamic filter's product_id need to be translated into correct id.
			$product_id = self::get_product_id( $maybe_product_id );
		} elseif ( $is_current_product_page && Conditions::is_rest_api_request() ) {
			/*
			 * $product global doesn't exist in REST request; thus get the fallback post id.
			 */
			$product_id = DynamicAssetsUtils::get_current_post_id();
		} else {
			// Besides two situation above, $product_id is current $args['product'].
			if ( false !== get_post_status( $maybe_product_id ) ) {
				$product_id = $maybe_product_id;
			} else {
				// Fallback to Latest product if saved product ID doesn't exist.
				$product_id = self::get_product_id( 'latest' );
			}
		}

		return $product_id;
	}

	/**
	 * Renders module templates based on specified actions and arguments. This function manages global context and
	 * ensures proper handling of the WooCommerce environment for different module templates.
	 *
	 * @since ??
	 *
	 * @param string $function_name The action or function name to process. It must be within the allowlist of
	 *                              supported functions.
	 * @param array  $args          Optional. An array of arguments to pass to the action or function. Default
	 *                              empty array.
	 * @param array  $overwrite     Optional. An array specifying which global variables should be temporarily
	 *                              overwritten (e.g., 'product', 'post', 'wp_query'). Default includes 'product'.
	 *
	 * @return string The generated output for the module template when applicable, or an empty string if the
	 * function cannot process the requested action.
	 */
	public static function render_module_template(
		string $function_name,
		array $args = [],
		array $overwrite = [ 'product' ]
	): string {
		// Bail early.
		if ( is_admin() && ! Conditions::is_rest_api_request() ) {
			return '';
		}

		// Check if passed function name is allowlisted or not.
		if ( ! in_array( $function_name, self::$_allowed_functions, true ) ) {
			return '';
		}

		// phpcs:disable WordPress.WP.GlobalVariablesOverride -- Overwrite global variables when rendering templates which are restored before this function exist.
		global $product, $post, $wp_query;

		/*
		 * TODO feat(D5, WooCommerce Imaegs Module): $defaults should be in D5 attribute, will be refactored in https://github.com/elegantthemes/Divi/issues/42668
		 */
		$defaults = [
			'product' => 'current',
		];

		$args               = wp_parse_args( $args, $defaults );
		$overwrite_global   = self::need_overwrite_global( $args['product'] );
		$overwrite_product  = in_array( 'product', $overwrite, true );
		$overwrite_post     = in_array( 'post', $overwrite, true );
		$overwrite_wp_query = in_array( 'wp_query', $overwrite, true );
		$is_tb              = et_builder_tb_enabled();
		$is_use_placeholder = $is_tb || is_et_pb_preview();

		if ( $is_use_placeholder ) {
			// global object needs to be set before output rendering. This needs to be performed on each
			// module template rendering instead of once for all module template rendering because some
			// module's template rendering uses `wp_reset_postdata()` which resets global query.
			et_theme_builder_wc_set_global_objects();
		} elseif ( $overwrite_global ) {
			$product_id = self::get_product_id_from_attributes( $args );

			if ( 'product' !== get_post_type( $product_id ) ) {
				// We are in a Theme Builder layout and the current post is not a product - use the latest one instead.
				$products = new WP_Query(
					[
						'post_type'      => 'product',
						'post_status'    => 'publish',
						'posts_per_page' => 1,
						'no_found_rows'  => true,
					]
				);

				if ( ! $products->have_posts() ) {
					return '';
				}

				$product_id = $products->posts[0]->ID;
			}

			// Overwrite product.
			if ( $overwrite_product ) {
				$original_product = $product;
				$product          = wc_get_product( $product_id );
			}

			// Overwrite post.
			if ( $overwrite_post ) {
				$original_post = $post;
				$post          = get_post( $product_id );
			}

			// Overwrite wp_query.
			if ( $overwrite_wp_query ) {
				$original_wp_query = $wp_query;
				$wp_query          = new WP_Query( [ 'p' => $product_id ] );
			}
		}

		ob_start();

		switch ( $function_name ) {
			case 'woocommerce_show_product_images':
				if ( is_a( $product, 'WC_Product' ) ) {
					// WC Images module needs to modify global variable's property. Thus it is performed
					// here instead at module's class since the $product global might be modified.
					$gallery_ids     = $product->get_gallery_image_ids();
					$image_id        = $product->get_image_id();
					$show_image      = 'on' === $args['show_product_image'];
					$show_gallery    = 'on' === $args['show_product_gallery'];
					$show_sale_badge = 'on' === $args['show_sale_badge'];

					// If featured image is disabled, replace it with first gallery image's id (if gallery
					// is enabled) or replaced it with empty string (if gallery is disabled as well).
					if ( ! $show_image ) {
						if ( $show_gallery && isset( $gallery_ids[0] ) ) {
							$product->set_image_id( $gallery_ids[0] );

							// Remove first image from the gallery because it'll be added as thumbnail and will be duplicated.
							unset( $gallery_ids[0] );
							$product->set_gallery_image_ids( $gallery_ids );
						} else {
							$product->set_image_id( '' );
						}
					}

					// Replaced gallery image ids with empty array.
					if ( ! $show_gallery ) {
						$product->set_gallery_image_ids( [] );
					}

					if ( $show_sale_badge && function_exists( 'woocommerce_show_product_sale_flash' ) ) {
						woocommerce_show_product_sale_flash();
					}

					// @phpcs:ignore Generic.PHP.ForbiddenFunctions.Found -- Using for consistency.
					call_user_func( $function_name );

					// Reset product's actual featured image id.
					if ( ! $show_image ) {
						$product->set_image_id( $image_id );
					}

					// Reset product's actual gallery image id.
					if ( ! $show_gallery ) {
						$product->set_gallery_image_ids( $gallery_ids );
					}
				}
				break;
			default:
				// Only whitelisted functions shall be allowed until this point of execution.
				if ( is_a( $product, 'WC_Product' ) ) {
					// @phpcs:ignore Generic.PHP.ForbiddenFunctions.Found -- Only whitelisted functions reach here.
					call_user_func( $function_name );
				}
		}

		$output = ob_get_clean();

		// Reset original product variable to global $product.
		if ( $is_use_placeholder ) {
			et_theme_builder_wc_reset_global_objects();
		} elseif ( $overwrite_global ) {
			// Reset $product global.
			if ( $overwrite_product ) {
				$product = $original_product;
			}

			// Reset post.
			if ( $overwrite_post ) {
				$post = $original_post;
			}

			// Reset wp_query.
			if ( $overwrite_wp_query ) {
				$wp_query = $original_wp_query;
			}
			// phpcs:enable WordPress.WP.GlobalVariablesOverride -- Enable global variable override check.
		}

		return $output;
	}

	/**
	 * Determines if WooCommerce's `$product` global needs to be overwritten.
	 *
	 * IMPORTANT: Ensure that the `$product` global is reset to its original state after use.
	 * Overwriting the global is necessary in specific scenarios to avoid using incorrect
	 * or stale product information.
	 *
	 * @since ??
	 *
	 * @param string $product_id Product ID to check against. Defaults to 'current', which means
	 *                           the current product page is being referenced.
	 *
	 * @return bool True if the global `$product` needs to be overwritten, false otherwise.
	 */
	public static function need_overwrite_global( string $product_id = 'current' ): bool {
		// Check if the provided product ID corresponds to the current product page.
		$is_current_product_page = 'current' === $product_id;

		/*
		 * The global `$product` variable needs to be overwritten in the following scenarios:
		 *
		 * 1. The specified `$product_id` is not for the current product page.
		 *
		 * 2. The current request is a WordPress REST API request.
		 *    This includes:
		 *    - Any REST requests made during AJAX calls (such as VB actions),
		 *      where the global `$product` is often inconsistent or incorrect.
		 *    - Special requests with `?rest_route=/` or prefixed with the REST API base URL.
		 */
		$need_overwrite_global = ! $is_current_product_page || Conditions::is_rest_api_request();

		// Return true if a global overwrite is needed, otherwise false.
		return $need_overwrite_global;
	}

	/**
	 * Determines whether given content has WooCommerce module inside it or not.
	 *
	 * This function is used to check if the content contains any WooCommerce-related blocks.
	 *
	 * This function is based on legacy `et_builder_has_woocommerce_module` function.
	 *
	 * @since ??
	 *
	 * @param string $content Content.
	 *
	 * @return bool
	 */
	public static function content_has_woocommerce_module( $content = '' ): bool {
		// regex test: https://regex101.com/r/QbPYzW/1.
		$woocommerce_block_regex = '/<!--\swp:divi\/woocommerce-.*?(?:\/-->|\/wp:divi\/woocommerce-.*\s--->)/s';
		$has_woocommerce_module  = preg_match( $woocommerce_block_regex, $content ) === 1;

		return apply_filters( 'divi_woocommerce_has_woocommerce_module', $has_woocommerce_module );
	}

	/**
	 * Sets the default product tabs for WooCommerce products.
	 *
	 * Defines the structure and properties of the default WooCommerce product tabs,
	 * including their titles, display priority, and callback functions for rendering
	 * the tab content. If Theme Builder is enabled, additional processing is performed
	 * to apply any customizations to the product tabs.
	 *
	 * Based on the legacy `get_default_product_tabs` function.
	 *
	 * @since ??
	 *
	 * @return array An array of default product tabs with their respective configurations.
	 */
	public static function set_default_product_tabs(): array {
		$tabs = [
			'description'            => [
				'title'    => esc_html__( 'Description', 'et_builder' ),
				'priority' => 10,
				'callback' => 'woocommerce_product_description_tab',
			],
			'additional_information' => [
				'title'    => esc_html__( 'Additional information', 'et_builder' ),
				'priority' => 20,
				'callback' => 'woocommerce_product_additional_information_tab',
			],
			'reviews'                => [
				'title'    => esc_html__( 'Reviews', 'et_builder' ),
				'priority' => 30,
				'callback' => 'comments_template',
			],
		];

		// Add custom tabs on default for theme builder.
		if ( et_builder_tb_enabled() ) {
			self::set_global_objects_for_theme_builder();

			$tabs = apply_filters( 'woocommerce_product_tabs', $tabs );

			self::reset_global_objects_for_theme_builder();
		}

		return $tabs;
	}

	/**
	 * Sets default product tabs options.
	 *
	 * Processes the default product tabs to generate an array of options
	 * containing tab names, values, and labels. Each option corresponds
	 * to a tab with a title attribute. Special handling is applied for the
	 * "reviews" tab to set its label.
	 *
	 * Based on the legacy `get_default_tab_options` function.
	 *
	 * @since ??
	 *
	 * @return array An associative array of default product tab options,
	 *               where each key represents a tab name and its value
	 *               contains value-label pairs. Returns an empty array
	 *               if no valid tabs are available.
	 */
	public static function set_default_product_tabs_options(): array {
		$tabs    = self::set_default_product_tabs();
		$options = [];

		foreach ( $tabs as $name => $tab ) {
			if ( ! isset( $tab['title'] ) ) {
				continue;
			}

			$options[ $name ] = [
				'value' => $name,
				'label' => 'reviews' === $name
					? esc_html__( 'Reviews', 'et_builder' )
					: esc_html( $tab['title'] ),
			];
		}

		return $options;
	}

	/**
	 * Sets global objects for the theme builder.
	 *
	 * Configures global variables and placeholders to ensure compatibility and rendering
	 * functionality within the theme builder. This includes preparing global `$product` and
	 * `$post` objects with correct placeholder or existing values.
	 *
	 * Based on the legacy `et_theme_builder_wc_set_global_objects` function.
	 *
	 * @since ??
	 *
	 * @param array $conditional_tags Associative array of conditional tags used for internal checks.
	 *                                Example keys:
	 *                                - 'is_tb' (bool): Whether the current request is related to the theme builder.
	 *
	 * @return void
	 */
	public static function set_global_objects_for_theme_builder( array $conditional_tags = [] ) {
		$is_tb              = $conditional_tags['is_tb'] ?? false;
		$is_use_placeholder = $is_tb || is_et_pb_preview();

		// Check if current request is theme builder (direct page / AJAX request).
		if ( ! et_builder_tb_enabled() && ! $is_use_placeholder ) {
			return;
		}

		// Global variable that affects WC module rendering.
		global $product, $post, $tb_original_product, $tb_original_post, $tb_wc_post, $tb_wc_product;

		// Making sure the correct comment template is loaded on WC tabs' review tab.
		// TODO feat(D5, WooCommerce Product Tabs Module): update the callback once we have the module for tabs in place [https://github.com/elegantthemes/Divi/issues/25756].
		add_filter( 'comments_template', [ 'ET_Builder_Module_Woocommerce_Tabs', 'comments_template_loader' ], 20 );

		// Force display related posts; technically sets all products as related.
		add_filter( 'woocommerce_product_related_posts_force_display', '__return_true' );

		// Make sure review's form is opened.
		add_filter( 'comments_open', '__return_true' );

		// Save original $post for reset later.
		$tb_original_post = $post;

		// Save original $product for reset later.
		$tb_original_product = $product;

		// If modified global existed, use it for efficiency.
		if ( ! is_null( $tb_wc_post ) && ! is_null( $tb_wc_product ) ) {
			// phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited -- Need to override the post with the theme builder post.
			$post    = $tb_wc_post;
			$product = $tb_wc_product;

			return;
		}

		// Get placeholders.
		$placeholders = et_theme_builder_wc_placeholders();

		if ( $is_use_placeholder ) {
			$placeholder_src = wc_placeholder_img_src( 'full' );
			$placeholder_id  = attachment_url_to_postid( $placeholder_src );

			if ( absint( $placeholder_id ) > 0 ) {
				$placeholders['gallery_image_ids'] = [ $placeholder_id ];
			}
		} else {
			$placeholders['gallery_image_ids'] = [];
		}

		// $post might be null if current request is computed callback (ie. WC gallery)
		if ( is_null( $post ) ) {
			// phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited -- Intentionally done.
			$post = new stdClass();
		}

		// Overwrite $post global.
		$post->post_title     = $placeholders['title'];
		$post->post_slug      = $placeholders['slug'];
		$post->post_excerpt   = $placeholders['short_description'];
		$post->post_content   = $placeholders['description'];
		$post->post_status    = $placeholders['status'];
		$post->comment_status = $placeholders['comment_status'];

		// Overwrite global $product.
		$product = new ET_Theme_Builder_Woocommerce_Product_Variable_Placeholder();

		// Set current post ID as product's ID. `ET_Theme_Builder_Woocommerce_Product_Variable_Placeholder`
		// handles all placeholder related value but product ID need to be manually set to match current
		// post's ID. This is especially needed when add-ons is used and accessing get_id() method.
		if ( isset( $post->ID ) ) {
			$product->set_id( $post->ID );
		}

		// Save modified global for later use.
		$tb_wc_post    = $post;
		$tb_wc_product = $product;
	}

	/**
	 * Resets global objects for use in the theme builder.
	 *
	 * Adjusts global variables and removes specific filters to prepare
	 * the environment for theme builder rendering or processing. This ensures
	 * proper behavior and compatibility when building or previewing themes.
	 *
	 * Based on the legacy `et_theme_builder_wc_reset_global_objects` function.
	 *
	 * @since ??
	 *
	 * @param array $conditional_tags Optional. An array of conditional tags to indicate
	 *                                the current context. Supports:
	 *                                - 'is_tb' (bool): Whether the current context is the
	 *                                  theme builder.
	 *
	 * @return void
	 */
	public static function reset_global_objects_for_theme_builder( array $conditional_tags = array() ) {
		$is_tb              = $conditional_tags['is_tb'] ?? false;
		$is_use_placeholder = $is_tb || is_et_pb_preview();

		// Check if current request is theme builder (direct page / AJAX request).
		if ( ! et_builder_tb_enabled() && ! $is_use_placeholder ) {
			return;
		}

		global $product, $post, $tb_original_product, $tb_original_post;

		// TODO feat(D5, WooCommerce Product Tabs Module): update the callback once we have the module for tabs in place [https://github.com/elegantthemes/Divi/issues/25756.
		remove_filter( 'comments_template', [ 'ET_Builder_Module_Woocommerce_Tabs', 'comments_template_loader' ], 20 );
		remove_filter( 'woocommerce_product_related_posts_force_display', '__return_true' );
		remove_filter( 'comments_open', '__return_true' );

		// phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited -- Need to override the post with the theme builder post.
		$post = $tb_original_post;

		// phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited -- Need to override the product with the theme builder product.
		$product = $tb_original_product;
	}

}
